# -*- coding: utf-8 -*-
"""
Deep Learning Land Cover Classification Toolbox
Author: ArcGIS Pro
Version: 2.0
Date: 2026
"""

import arcpy
import os
import sys
import urllib.request
import zipfile
import subprocess
import shutil
import urllib.error
import json
import time
import warnings
import platform
import traceback
import importlib
import tempfile
from pathlib import Path
import uuid
import urllib.parse
import getpass
import random
import ctypes
import base64
from datetime import datetime, timezone

warnings.filterwarnings('ignore')

def get_python_exe():
    """Get the correct python executable path"""
    # In ArcGIS Pro, sys.executable points to ArcGISPro.exe
    # We need the actual python.exe in the environment
    python_exe = os.path.join(sys.exec_prefix, 'python.exe')
    
    # Fallback check
    if not os.path.exists(python_exe):
        python_exe = os.path.join(sys.exec_prefix, 'Scripts', 'python.exe')
    
    if not os.path.exists(python_exe):
        # Last resort: use sys.executable (but likely wrong)
        return sys.executable
        
    return python_exe

def get_hide_window_kwargs():
    """Get arguments to hide subprocess window"""
    kwargs = {}
    if platform.system() == 'Windows':
        startupinfo = subprocess.STARTUPINFO()
        startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW
        startupinfo.wShowWindow = subprocess.SW_HIDE
        kwargs['startupinfo'] = startupinfo
        kwargs['creationflags'] = 0x08000000 # CREATE_NO_WINDOW
    return kwargs

class Toolbox(object):
    def __init__(self):
        """Define the toolbox (the name of the toolbox is the name of the
        .pyt file)."""
        self.label = "Deep Learning Land Cover Toolbox"
        self.alias = "DLLandCover"
        self.description = "Toolbox for Land Cover Classification using Deep Learning"

        # List of tool classes associated with this toolbox
        self.tools = [CheckLicenseStatus, InstallLibraries, ManageLandsat8, ClassifyLandCover]

# ===================================================================================
#  SUPABASE LICENSE GUARD & AUTO-UPDATE SYSTEM
# ===================================================================================
CURRENT_VERSION = "2.0.0"

class UpdateManager:
    @staticmethod
    def is_new_version(remote, current):
        try:
            r_parts = [int(x) for x in remote.split('.')]
            c_parts = [int(x) for x in current.split('.')]
            return r_parts > c_parts
        except: return remote != current

    @staticmethod
    def generate_popup_script(title, message, type='info', data=None):
        try:
            py_exe = get_python_exe()
            
            script_header = """
import tkinter as tk
import sys
import base64
from io import BytesIO

def show():
    try:
        root = tk.Tk()
        root.withdraw()
        
        COLOR_PRIMARY = '#10B981'
        COLOR_HOVER   = '#059669'
        COLOR_BG      = '#F9FAFB'
        COLOR_CARD    = '#FFFFFF'
        COLOR_TEXT    = '#1F2937'
        COLOR_SUBTEXT = '#6B7280'

        dialog = tk.Toplevel(root)
        dialog.title('Deep Learning Tools Notification')
        dialog.overrideredirect(True)
        dialog.overrideredirect(False)
        
        w = 420
        h = 520
        if TYPE == 'update': h = 420
        if TYPE == 'license_expired': h = 550
        
        sw = root.winfo_screenwidth()
        sh = root.winfo_screenheight()
        x = (sw - w) // 2
        y = (sh - h) // 2
        dialog.geometry(f'{w}x{h}+{x}+{y}')
        dialog.resizable(False, False)
        dialog.configure(bg=COLOR_BG)
        
        dialog.lift()
        dialog.attributes('-topmost',True)
        dialog.after_idle(root.attributes,'-topmost',False)

        card = tk.Frame(dialog, bg=COLOR_CARD)
        card.pack(fill="both", expand=True)

        header = tk.Frame(card, bg=COLOR_PRIMARY, pady=20, padx=15)
        header.pack(fill="x", side="top")
        
        left_col = tk.Frame(header, bg=COLOR_PRIMARY)
        left_col.pack(side="left", anchor="w")

        right_col = tk.Frame(header, bg=COLOR_PRIMARY)
        right_col.pack(side="left", fill="both", expand=True, padx=(10, 0))

        LOGO_B64 = "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"
        FAVICON_B64 = "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"

        try:
            fav_data = base64.b64decode(FAVICON_B64)
            icon_img = tk.PhotoImage(data=fav_data)
            dialog.iconphoto(False, icon_img)
        except: pass

        try:
            img_data = base64.b64decode(LOGO_B64)
            logo_img = tk.PhotoImage(data=img_data)
            logo_lbl = tk.Label(left_col, image=logo_img, bg=COLOR_PRIMARY)
            logo_lbl.image = logo_img
            logo_lbl.pack(anchor="w")
        except: pass
        
        lbl_title = tk.Label(right_col, text="TITLE_PLACEHOLDER", font=("Segoe UI", 18, "bold"), bg=COLOR_PRIMARY, fg="white", wraplength=250, justify="center")
        lbl_title.pack(anchor="center", pady=(5, 0))
        
        lbl_msg = tk.Label(right_col, text="MESSAGE_PLACEHOLDER", font=("Segoe UI", 11), bg=COLOR_PRIMARY, fg="#ECFDF5", wraplength=250, justify="center")
        lbl_msg.pack(anchor="center", pady=(0, 5))

        content_frame = tk.Frame(card, bg=COLOR_CARD)
        content_frame.pack(fill="both", expand=False, padx=20, pady=20)
"""
            
            script_body = ""
            
            if type in ['license', 'license_expired']:
                if data:
                    app_name = data.get('app_name', 'deep_learning_tools')
                    status = data.get('status', 'Unknown')
                    tier = data.get('tier', 'Unknown')
                    expiry = data.get('expiry_date', 'N/A')
                    days_left = data.get('days_left', 'N/A')
                    machine_id = data.get('machine_id', 'N/A')
                    admin_contact = "Ardi Abu Ridho (+62 822-5476-0769)"

                    status_upper = status.upper()
                    tier_upper = tier.upper()
                    
                    if status == 'expired':
                        status_color = '#EF4444'
                        bg_color = '#FEF2F2'
                    else:
                        status_color = '#10B981'
                        bg_color = '#F0FDF4'
                        
                    script_body = f"""
        info_frame = tk.Frame(content_frame, bg='{{bg_color}}')
        info_frame.pack(pady=10, fill='x', ipadx=10, ipady=10)
        tk.Label(info_frame, text='STATUS LISENSI (DEEP LEARNING)', font=('Segoe UI', 11, 'bold'), bg='{{bg_color}}', fg='{{status_color}}').pack(pady=(0,8))
        
        info_grid = tk.Frame(info_frame, bg='{{bg_color}}')
        info_grid.pack()
        
        tk.Label(info_grid, text='Aplikasi :', font=('Segoe UI', 10, 'bold'), bg='{{bg_color}}', fg=COLOR_TEXT).grid(row=0, column=0, sticky='e', padx=5, pady=2)
        tk.Label(info_grid, text='{{app_name}}', font=('Segoe UI', 10), bg='{{bg_color}}', fg=COLOR_TEXT).grid(row=0, column=1, sticky='w', pady=2)
        
        tk.Label(info_grid, text='ID Mesin :', font=('Segoe UI', 10, 'bold'), bg='{{bg_color}}', fg=COLOR_TEXT).grid(row=1, column=0, sticky='e', padx=5, pady=2)
        tk.Label(info_grid, text='{{machine_id}}', font=('Segoe UI', 10), bg='{{bg_color}}', fg=COLOR_TEXT).grid(row=1, column=1, sticky='w', pady=2)
        
        tk.Label(info_grid, text='Status :', font=('Segoe UI', 10, 'bold'), bg='{{bg_color}}', fg=COLOR_TEXT).grid(row=2, column=0, sticky='e', padx=5, pady=2)
        tk.Label(info_grid, text='{{status_upper}}', font=('Segoe UI', 10, 'bold'), bg='{{bg_color}}', fg='{{status_color}}').grid(row=2, column=1, sticky='w', pady=2)
        
        tk.Label(info_grid, text='Tipe Paket :', font=('Segoe UI', 10, 'bold'), bg='{{bg_color}}', fg=COLOR_TEXT).grid(row=3, column=0, sticky='e', padx=5, pady=2)
        tk.Label(info_grid, text='{{tier_upper}}', font=('Segoe UI', 10), bg='{{bg_color}}', fg=COLOR_TEXT).grid(row=3, column=1, sticky='w', pady=2)
        
        tk.Label(info_grid, text='Masa Aktif :', font=('Segoe UI', 10, 'bold'), bg='{{bg_color}}', fg=COLOR_TEXT).grid(row=4, column=0, sticky='e', padx=5, pady=2)
        expiry_display = '{{expiry}}' if '{{expiry}}' == 'Lifetime' else '{{expiry}} (Sisa {{days_left}} hari)'
        tk.Label(info_grid, text=expiry_display, font=('Segoe UI', 10), bg='{{bg_color}}', fg=COLOR_TEXT).grid(row=4, column=1, sticky='w', pady=2)
        
        tk.Label(content_frame, text='Hubungi Admin jika ada kendala:', font=('Segoe UI', 9), bg=COLOR_CARD, fg=COLOR_SUBTEXT).pack(pady=(15,2))
        tk.Label(content_frame, text='{{admin_contact}}', font=('Segoe UI', 10, 'bold'), bg=COLOR_CARD, fg='#0EA5E9').pack()
""".format(app_name=app_name, machine_id=machine_id, status_upper=status_upper, status_color=status_color, 
            tier_upper=tier_upper, expiry=expiry, days_left=days_left, bg_color=bg_color, admin_contact=admin_contact)

            script_footer = """
        footer = tk.Frame(dialog, bg='#F3F4F6', pady=15, padx=20)
        footer.pack(fill='x', side='bottom')
        
        def on_ok():
            dialog.destroy()
            root.destroy()
            sys.exit(0)
            
        tk.Button(footer, text='OK', command=on_ok, bg=COLOR_PRIMARY, fg='white', font=('Segoe UI', 12, 'bold'), relief='flat', width=12).pack(side='right')
        root.mainloop()
    except: sys.exit(1)

if __name__ == "__main__": show()
"""
            
            type_def = f"TYPE = '{type}'\n"
            final_script = type_def + script_header.replace("TITLE_PLACEHOLDER", title).replace("MESSAGE_PLACEHOLDER", message) + script_body + script_footer
            
            script_path = os.path.join(tempfile.gettempdir(), f"dl_popup_{random.randint(100,999)}.py")
            with open(script_path, "w") as f: f.write(final_script)
            
            subprocess.Popen([py_exe, script_path], **get_hide_window_kwargs())
            time.sleep(1)
            try: os.remove(script_path)
            except: pass
            return True
        except: return False

class SupabaseGuard:
    SUPABASE_URL = "https://mbfzmvlivyuajmxrecne.supabase.co"
    SUPABASE_KEY = "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6Im1iZnptdmxpdnl1YWpteHJlY25lIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NjU0NzgwODQsImV4cCI6MjA4MTA1NDA4NH0.sZltNY30ww2Hb_oopiVDcvXnZRehWRvK2jZZU5MO64s"
    TABLE_NAME = "licenses"
    ADMIN_CONTACT = "Ardi Abu Ridho (+62 822-5476-0769)"
    
    @staticmethod
    def get_machine_id():
        try:
            # Try Windows UUID
            cmd = 'wmic csproduct get uuid'
            res = subprocess.check_output(cmd, shell=True, **get_hide_window_kwargs()).decode()
            uid = res.split('\n')[1].strip()
            if uid: return uid.replace('-', '')[:12]
        except: pass
        return hex(uuid.getnode())[2:].upper()

    @staticmethod
    def make_request(url, method="GET", data=None):
        headers = {
            "apikey": SupabaseGuard.SUPABASE_KEY,
            "Authorization": f"Bearer {SupabaseGuard.SUPABASE_KEY}",
            "Content-Type": "application/json",
            "Prefer": "return=representation"
        }
        try:
            req_data = json.dumps(data).encode('utf-8') if data else None
            req = urllib.request.Request(url, data=req_data, headers=headers, method=method)
            with urllib.request.urlopen(req, timeout=10) as response:
                if response.status in [200, 201, 204]:
                    return json.loads(response.read().decode()) or []
        except: return None

    @staticmethod
    def check_license(app_name="deep_learning_landcover"):
        mid = SupabaseGuard.get_machine_id()
        base_url = f"{SupabaseGuard.SUPABASE_URL}/rest/v1/{SupabaseGuard.TABLE_NAME}"
        
        # 1. Check Existing
        query = urllib.parse.urlencode({"machine_id": f"eq.{mid}", "app_name": f"eq.{app_name}"})
        data = SupabaseGuard.make_request(f"{base_url}?{query}", "GET")
        
        # 2. Register New (Auto Trial)
        if not data:
            payload = {
                "machine_id": mid,
                "app_name": app_name,
                "last_check": datetime.now(timezone.utc).isoformat(),
                "tier": "pro",
                "client_name": f"{platform.node()} ({getpass.getuser()})",
                "version": CURRENT_VERSION
            }
            
            data = SupabaseGuard.make_request(base_url, "POST", payload)
            if not data: return False, "Gagal registrasi trial server.", {}
        else:
            # 2. Update Existing (Tracking Last Check)
            try:
                mid_query = f"eq.{mid}"
                app_query = f"eq.{app_name}"
                patch_url = f"{base_url}?machine_id={mid_query}&app_name={app_query}"
                
                update_payload = {
                    "last_check": datetime.now(timezone.utc).isoformat(),
                    "version": CURRENT_VERSION,
                    "client_name": f"{platform.node()} ({getpass.getuser()})"
                }
                SupabaseGuard.make_request(patch_url, "PATCH", update_payload)
            except: pass # Non-fatal if update fails
        
        rec = data[0]
        status = rec.get('status', 'trial').lower()
        expiry = rec.get('expiry_date')
        
        # 3. Check Expiry
        days_left = 999
        if expiry and expiry != 'Lifetime':
            try:
                exp_date = datetime.strptime(expiry, "%Y-%m-%d")
                days_left = (exp_date - datetime.now()).days
                if days_left < 0: status = 'expired'
            except: pass
        
        info = {**rec, 'days_left': days_left}
        
        if status in ['banned', 'expired'] and rec.get('tier') != 'free':
            msg = f"⛔ Lisensi {status.upper()}. Hubungi {SupabaseGuard.ADMIN_CONTACT}"
            UpdateManager.generate_popup_script("Akses Ditolak", msg, data=info)
            return False, msg, info
            
        return True, "Valid", info

class CheckLicenseStatus(object):
    def __init__(self):
        self.label = "01. Cek Status Lisensi (Check License Status)"
        self.category = "00. System & Licensing"
        self.canRunInBackground = False
    def getParameterInfo(self):
        # Info Parameter (Display Only)
        param0 = arcpy.Parameter(
            displayName="[INFO] Petunjuk Penggunaan",
            name="info_display",
            datatype="GPString",
            parameterType="Optional",
            direction="Input")
        param0.value = "Klik tombol RUN di bawah untuk memulai pemeriksaan lisensi"
        return [param0]

    def execute(self, parameters, messages):
        messages.addMessage("Memeriksa status lisensi...")
        valid, msg, info = SupabaseGuard.check_license()
        
        status = info.get('status', 'Unknown').upper()
        tier = info.get('tier', 'Unknown').upper()
        expiry = info.get('expiry_date', 'Lifetime')
        
        messages.addMessage("============================================================")
        messages.addMessage(" STATUS LISENSI")
        messages.addMessage("============================================================")
        messages.addMessage(f" Aplikasi     : {info.get('app_name', 'Unknown')}")
        messages.addMessage(f" ID Mesin     : {info.get('machine_id')}")
        messages.addMessage(f" Status       : {status}")
        messages.addMessage(f" Tipe Paket   : {tier}")
        messages.addMessage(f" Masa Aktif   : {expiry}")
        messages.addMessage("")
        messages.addMessage(" Hubungi Admin jika ada kendala:")
        messages.addMessage(f" {SupabaseGuard.ADMIN_CONTACT}")
        messages.addMessage("============================================================")
        messages.addMessage("")
        
        if status == 'TRIAL':
            try:
                # Calculate remaining days
                if expiry != 'Lifetime':
                    exp_date = datetime.fromisoformat(expiry.replace('Z', '+00:00'))
                    now = datetime.now(timezone.utc)
                    delta = (exp_date - now).days
                    messages.addMessage(f"INFO: Trial period remaining: {delta} days.")
            except: pass
        
        messages.addMessage("[INFO] Tips: Klik Kanan Toolbox -> Refresh untuk mengecek update terbaru.")
        messages.addMessage("")
        messages.addMessage("============================================================")
        messages.addMessage("[OK] Selesai memeriksa lisensi.")
        messages.addMessage("============================================================")
        
        # Determine Popup Type
        popup_type = 'license'
        if status == 'EXPIRED': popup_type = 'license_expired'
        
        UpdateManager.generate_popup_script("Info Lisensi", "Detail status lisensi Anda", type=popup_type, data=info)

class InstallLibraries(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "02. Install Deep Learning Libraries (Otomatis)"
        self.category = "00. System & Licensing"
        self.description = """Otomatis mendeteksi dan memperbaiki Deep Learning Libraries.
        Tool ini akan:
        1. Mendeteksi environment Python yang aktif
        2. Memperbaiki dependencies yang rusak/missing
        3. Menginstal PyTorch, fastai, dan dependencies lainnya
        4. Memperbaiki environment yang bermasalah"""
        self.canRunInBackground = False

    def getParameterInfo(self):
        """Define parameter definitions"""
        
        # Param 0: Repair Mode
        param0 = arcpy.Parameter(
            displayName="Mode Perbaikan",
            name="repair_mode",
            datatype="GPString",
            parameterType="Required",
            direction="Input"
        )
        param0.filter.type = "ValueList"
        param0.filter.list = ["Auto Detect & Fix", "Force Reinstall All", "Check Only"]
        param0.value = "Auto Detect & Fix"
        
        # Param 1: Use GPU
        param1 = arcpy.Parameter(
            displayName="Gunakan GPU (CUDA)",
            name="use_gpu",
            datatype="GPBoolean",
            parameterType="Optional",
            direction="Input"
        )
        param1.value = True
        
        # Param 2: Download Folder
        param2 = arcpy.Parameter(
            displayName="Folder Download",
            name="download_folder",
            datatype="DEFolder",
            parameterType="Optional",
            direction="Input"
        )
        # Default ke folder Downloads user
        default_dir = os.path.join(os.environ.get('USERPROFILE', 'C:\\'), 'Downloads')
        if os.path.exists(default_dir):
            param2.value = default_dir
        
        return [param0, param1, param2]

    def isLicensed(self):
        return True

    def updateParameters(self, parameters):
        return

    def updateMessages(self, parameters):
        return

    def execute(self, parameters, messages):
        """Execute the tool"""
        repair_mode = parameters[0].valueAsText
        use_gpu = parameters[1].value
        download_folder = parameters[2].valueAsText
        
        # Check License
        SupabaseGuard.check_license()
        
        messages.addMessage("=" * 60)
        messages.addMessage("DEEP LEARNING LIBRARIES INSTALLER & FIXER")
        messages.addMessage("=" * 60)
        
        # ====================================================================
        # 1. SYSTEM DIAGNOSIS
        # ====================================================================
        messages.addMessage("\n[1] DIAGNOSIS SYSTEM...")
        
        # Get system info
        sys_info = self.get_system_info()
        for key, value in sys_info.items():
            messages.addMessage(f"  {key}: {value}")
        
        # Check Python environment
        env_info = self.check_python_environment()
        messages.addMessage(f"\n  Environment: {env_info['env_name']}")
        messages.addMessage(f"  Python: {env_info['python_version']}")
        messages.addMessage(f"  ArcGIS Pro: {env_info['arcgis_version']}")
        messages.addMessage(f"  Path: {env_info['env_path']}")
        
        # Check for incompatible environment
        if env_info['is_problematic']:
            messages.addWarningMessage(f"  ⚠️ Environment bermasalah: {env_info['issue']}")
        
        # ====================================================================
        # 2. CHECK DEEP LEARNING DEPENDENCIES
        # ====================================================================
        messages.addMessage("\n[2] CHECKING DEPENDENCIES...")
        
        deps_status = self.check_dependencies()
        missing_deps = []
        for dep, status in deps_status.items():
            if status['installed']:
                messages.addMessage(f"  ✓ {dep}: {status['version']}")
            else:
                messages.addMessage(f"  ✗ {dep}: MISSING")
                missing_deps.append(dep)
        
        # ====================================================================
        # 3. DECIDE ACTION BASED ON MODE
        # ====================================================================
        if repair_mode == "Check Only":
            messages.addMessage("\n[3] CHECK ONLY MODE - Tidak melakukan instalasi")
            messages.addMessage(f"   Total dependencies: {len(deps_status)}")
            messages.addMessage(f"   Terinstal: {len(deps_status) - len(missing_deps)}")
            messages.addMessage(f"   Missing: {len(missing_deps)}")
            if missing_deps:
                messages.addWarningMessage(f"   Dependencies missing: {', '.join(missing_deps)}")
            return
        
        # ====================================================================
        # 4. FIX ENVIRONMENT ISSUES
        # ====================================================================
        messages.addMessage("\n[3] FIXING ENVIRONMENT ISSUES...")
        
        # Fix 1: Check if running in correct environment
        if env_info['is_problematic']:
            fix_result = self.fix_environment_issue(env_info, messages)
            if not fix_result:
                messages.addErrorMessage("Gagal memperbaiki environment. Silakan switch ke 'arcgispro-py3' manual.")
                return
        
        # Fix 2: Repair pip
        messages.addMessage("  Memperbaiki pip...")
        self.repair_pip()
        
        # Fix 3: Upgrade setuptools and wheel
        messages.addMessage("  Memperbarui setuptools dan wheel...")
        self.upgrade_build_tools()
        
        # ====================================================================
        # 5. INSTALL/FIX DEPENDENCIES
        # ====================================================================
        messages.addMessage("\n[4] INSTALLING DEPENDENCIES...")
        
        # Determine CUDA version
        cuda_version = "cu118"  # Default for ArcGIS Pro
        if use_gpu:
            cuda_available = self.check_cuda_available()
            if cuda_available:
                messages.addMessage("  ✓ CUDA terdeteksi, menggunakan GPU version")
            else:
                messages.addMessage("  ⚠️ CUDA tidak terdeteksi, menggunakan CPU version")
                cuda_version = "cpu"
        
        # Install PyTorch
        messages.addMessage(f"  Menginstal PyTorch (CUDA: {cuda_version})...")
        torch_success = self.install_pytorch(cuda_version, messages)
        
        if not torch_success and repair_mode == "Force Reinstall All":
            messages.addMessage("  Mencoba metode alternatif...")
            torch_success = self.install_pytorch_alternative(cuda_version, messages)
        
        # Install fastai
        if torch_success:
            messages.addMessage("  Menginstal fastai...")
            fastai_success = self.install_fastai(messages)
        else:
            messages.addWarningMessage("  ⚠️ Melewati fastai karena PyTorch gagal")
            fastai_success = False
        
        # Install other dependencies
        messages.addMessage("  Menginstal dependencies lainnya...")
        self.install_other_dependencies(messages)
        
        # ====================================================================
        # 6. VERIFICATION
        # ====================================================================
        messages.addMessage("\n[5] VERIFIKASI INSTALASI...")
        
        # Re-check dependencies
        final_check = self.check_dependencies()
        all_ok = all(status['installed'] for status in final_check.values())
        
        if all_ok:
            messages.addMessage("  ✓ SEMUA DEPENDENCIES TERINSTAL DENGAN BAIK!")
        else:
            messages.addWarningMessage("  ⚠️ Beberapa dependencies masih bermasalah:")
            for dep, status in final_check.items():
                if not status['installed']:
                    messages.addWarningMessage(f"    - {dep}")
        
        # Test imports
        messages.addMessage("\n  Testing imports...")
        test_results = self.test_imports()
        for module, success in test_results.items():
            if success:
                messages.addMessage(f"    ✓ {module}: OK")
            else:
                messages.addErrorMessage(f"    ✗ {module}: GAGAL")
        
        # ====================================================================
        # 7. FINAL MESSAGES
        # ====================================================================
        messages.addMessage("\n" + "=" * 60)
        messages.addMessage("INSTALASI SELESAI!")
        messages.addMessage("=" * 60)
        
        if all_ok and all(test_results.values()):
            messages.addMessage("\n✅ SEMUA DEPENDENCIES SIAP DIGUNAKAN!")
            messages.addMessage("Anda sekarang dapat menjalankan tool klasifikasi.")
        else:
            messages.addWarningMessage("\n⚠️ ADA MASALAH YANG PERLU DIPERBAIKI:")
            messages.addWarningMessage("1. Pastikan environment yang benar ('arcgispro-py3')")
            messages.addWarningMessage("2. Jalankan tool ini lagi dengan mode 'Force Reinstall All'")
            messages.addWarningMessage("3. Atau install manual via Command Prompt:")
            messages.addMessage("\nPerintah manual:")
            messages.addMessage(f'"{sys.executable}" -m pip install torch==2.1.2 torchvision==0.16.2')
            messages.addMessage(f'"{sys.executable}" -m pip install fastai==2.7.13')
            messages.addMessage(f'"{sys.executable}" -m pip install cryptography pillow')
        
        messages.addMessage("\nTips:")
        messages.addMessage("- Gunakan environment 'arcgispro-py3' untuk kompatibilitas terbaik")
        messages.addMessage("- Restart ArcGIS Pro jika mengalami masalah")
        messages.addMessage("- Pastikan GPU drivers up to date untuk performa terbaik")
        
        return

    # =========================================================================
    # HELPER METHODS
    # =========================================================================
    
    def get_system_info(self):
        """Get system information"""
        info = {
            "OS": platform.system() + " " + platform.release(),
            "Architecture": platform.architecture()[0],
            "Processor": platform.processor(),
            "Python Executable": sys.executable,
            "Python Path": sys.prefix
        }
        return info
    
    def check_python_environment(self):
        """Check Python environment status"""
        env_path = sys.prefix
        python_version = sys.version.split()[0]
        
        # Get ArcGIS Pro version
        try:
            version_info = arcpy.GetInstallInfo()
            arcgis_version = version_info.get('Version', 'Unknown')
        except:
            arcgis_version = 'Unknown'
        
        # Detect environment name
        env_name = "Unknown"
        is_problematic = False
        issue = ""
        
        if "geo-geemap" in env_path.lower():
            env_name = "geo-geemap"
            is_problematic = True
            issue = "Environment custom, mungkin Python 3.13 (tidak kompatibel)"
        elif "arcgispro-py3" in env_path.lower():
            env_name = "arcgispro-py3"
        elif "pro-py3" in env_path.lower():
            env_name = "arcgispro-py3"
        else:
            env_name = os.path.basename(env_path)
        
        # Check Python version compatibility
        if "3.13" in python_version:
            # Python 3.13 is new, but we must try to support it if it's the only option.
            is_problematic = False # Allow execution
            issue = "Python 3.13 terdeteksi (Experimental)"
        
        return {
            'env_name': env_name,
            'env_path': env_path,
            'python_version': python_version,
            'arcgis_version': arcgis_version,
            'is_problematic': is_problematic,
            'issue': issue
        }
    
    def check_dependencies(self):
        """Check if deep learning dependencies are installed"""
        dependencies = {
            'torch': {'pip_name': 'torch', 'min_version': '2.0.0'},
            'torchvision': {'pip_name': 'torchvision', 'min_version': '0.15.0'},
            'fastai': {'pip_name': 'fastai', 'min_version': '2.7.0'},
            'cryptography': {'pip_name': 'cryptography', 'min_version': '40.0.0'},
            'Pillow': {'pip_name': 'Pillow', 'min_version': '9.0.0'},
            'numpy': {'pip_name': 'numpy', 'min_version': '1.21.0'},
            'pandas': {'pip_name': 'pandas', 'min_version': '1.3.0'},
        }
        
        results = {}
        for dep_name, dep_info in dependencies.items():
            try:
                # Try to import
                if dep_name == 'torch':
                    import torch
                    version = torch.__version__
                elif dep_name == 'torchvision':
                    import torchvision
                    version = torchvision.__version__
                elif dep_name == 'fastai':
                    import fastai
                    version = fastai.__version__
                elif dep_name == 'cryptography':
                    import cryptography
                    version = cryptography.__version__
                elif dep_name == 'Pillow':
                    from PIL import __version__ as pil_version
                    version = pil_version
                elif dep_name == 'numpy':
                    import numpy
                    version = numpy.__version__
                elif dep_name == 'pandas':
                    import pandas
                    version = pandas.__version__
                
                results[dep_name] = {
                    'installed': True,
                    'version': version
                }
            except ImportError:
                results[dep_name] = {
                    'installed': False,
                    'version': 'Not installed'
                }
            except Exception as e:
                results[dep_name] = {
                    'installed': False,
                    'version': f'Error: {str(e)}'
                }
        
        return results
    
    def fix_environment_issue(self, env_info, messages):
        """Try to fix environment issues"""
        # We try to proceed even if flagged, unless it's critical
        return True
    

    def repair_pip(self):
        """Repair pip installation"""
        try:
            python_exe = get_python_exe()
            subprocess.run([
                python_exe, "-m", "pip", "install", "--upgrade", "pip",
                "--trusted-host", "pypi.org",
                "--trusted-host", "files.pythonhosted.org"
            ], check=True, capture_output=True, timeout=60, **get_hide_window_kwargs())
            return True
        except:
            return False
    
    def upgrade_build_tools(self):
        """Upgrade setuptools and wheel"""
        try:
            python_exe = get_python_exe()
            subprocess.run([
                python_exe, "-m", "pip", "install", "--upgrade",
                "setuptools", "wheel",
                "--trusted-host", "pypi.org",
                "--trusted-host", "files.pythonhosted.org"
            ], check=True, capture_output=True, timeout=60, **self.get_hide_window_kwargs())
            return True
        except:
            return False
    
    def check_cuda_available(self):
        """Check if CUDA is available"""
        try:
            # Try to import torch and check CUDA
            import torch
            return torch.cuda.is_available()
        except:
            return False
    
    def install_pytorch(self, cuda_version, messages):
        """Install PyTorch with specific CUDA version"""
        try:
            python_exe = get_python_exe()
            
            # Check if Python 3.13
            is_py313 = "3.13" in sys.version
            
            # Determine PyTorch URL and Version based on Python Version
            # This ensures compatibility with older ArcGIS Pro versions (2.x, 3.0-3.5)
            # Py 3.6/3.7 (Pro 2.x) -> Torch 1.10
            # Py 3.8-3.9 (Pro 3.0) -> Torch 1.13 or 2.0
            # Py 3.10+ (Pro 3.1+)  -> Torch 2.x
            
            py_ver = sys.version_info
            
            if cuda_version == "cu118":
                torch_url = "https://download.pytorch.org/whl/cu118"
            elif cuda_version == "cpu":
                torch_url = "https://download.pytorch.org/whl/cpu"
            else:
                torch_url = "https://download.pytorch.org/whl/cu118"

            if py_ver.major == 3 and py_ver.minor <= 7:
                 # Legacy Pro 2.x (Python <= 3.7)
                torch_spec = "torch==1.10.0 torchvision==0.11.1" # Last stable for Py3.6/3.7
                if cuda_version == "cu118": 
                    # CUDA 11.8 too new for Torch 1.10, switch URL to cu113
                    torch_url = "https://download.pytorch.org/whl/cu113"
            elif py_ver.major == 3 and py_ver.minor < 13:
                # Mainstream Pro 3.x (Python 3.8 - 3.12)
                # Let pip install a compatible 2.x version, but prefer 2.1.2 if precise control needed
                # But 'torch torchvision' is safer for distribution as it picks best for detected OS/Py
                torch_spec = "torch==2.1.2 torchvision==0.16.2"
            else:
                # Bleeding Edge / Python 3.13+
                torch_spec = "torch torchvision"
            
            # Install PyTorch
            cmd = [
                python_exe, "-m", "pip", "install",
                *torch_spec.split(),
                "--index-url", torch_url,
                "--extra-index-url", "https://pypi.org/simple",
                "--trusted-host", "pypi.org",
                "--trusted-host", "files.pythonhosted.org",
                "--force-reinstall"
            ]
            
            messages.addMessage(f"  Menjalankan: {' '.join(cmd)}")
            
            result = subprocess.run(
                cmd,
                capture_output=True,
                text=True,
                timeout=600,  # 10 minutes timeout
                **get_hide_window_kwargs()
            )
            
            if result.returncode == 0:
                messages.addMessage("  ✓ PyTorch berhasil diinstal")
                return True
            else:
                stderr_lower = result.stderr.lower()
                if "access is denied" in stderr_lower or "permission denied" in stderr_lower:
                    messages.addErrorMessage("\n⛔ PERMISSION ERROR: Environment Anda terkunci/read-only.")
                    messages.addErrorMessage("Solusi:")
                    messages.addErrorMessage("1. Tutup ArcGIS Pro.")
                    messages.addErrorMessage("2. Klik Kanan ArcGIS Pro -> Run as Administrator. Jalankan tool lagi.")
                    messages.addErrorMessage("3. ATAU: Clone Environment di Package Manager.")
                    return False
                else:
                    messages.addWarningMessage(f"  ⚠️ Gagal instal PyTorch: {result.stderr[:200]}")
                    return False
                
        except subprocess.TimeoutExpired:
            messages.addWarningMessage("  ⚠️ Timeout saat instal PyTorch")
            return False
        except Exception as e:
            messages.addWarningMessage(f"  ⚠️ Error instal PyTorch: {str(e)}")
            return False
    
    def install_pytorch_alternative(self, cuda_version, messages):
        """Alternative PyTorch installation method"""
        try:
            python_exe = get_python_exe()
            
            # Try without specific index URL
            cmd = [
                python_exe, "-m", "pip", "install",
                "torch==2.1.2", "torchvision==0.16.2",
                "--extra-index-url", "https://download.pytorch.org/whl/cu118",
                "--trusted-host", "pypi.org",
                "--trusted-host", "download.pytorch.org",
                "--force-reinstall"
            ]
            
            messages.addMessage("  Mencoba metode alternatif...")
            
            result = subprocess.run(
                cmd,
                capture_output=True,
                text=True,
                timeout=300,
                **get_hide_window_kwargs()
            )
            
            if result.returncode == 0:
                messages.addMessage("  ✓ PyTorch berhasil diinstal (metode alternatif)")
                return True
            else:
                messages.addWarningMessage("  ⚠️ Gagal dengan metode alternatif")
                return False
                
        except Exception as e:
            messages.addWarningMessage(f"  ⚠️ Error metode alternatif: {str(e)}")
            return False
    
    def install_fastai(self, messages):
        """Install fastai"""
        try:
            python_exe = get_python_exe()
            
            cmd = [
                python_exe, "-m", "pip", "install",
                "fastai==2.7.13",
                "--trusted-host", "pypi.org",
                "--trusted-host", "files.pythonhosted.org"
            ]
            
            result = subprocess.run(
                cmd,
                capture_output=True,
                text=True,
                timeout=180,  # 3 minutes timeout
                **get_hide_window_kwargs()
            )
            
            if result.returncode == 0:
                messages.addMessage("  ✓ fastai berhasil diinstal")
                return True
            else:
                messages.addWarningMessage(f"  ⚠️ Gagal instal fastai: {result.stderr[:200]}")
                return False
                
        except Exception as e:
            messages.addWarningMessage(f"  ⚠️ Error instal fastai: {str(e)}")
            return False
    
    def install_other_dependencies(self, messages):
        """Install other required dependencies"""
        try:
            python_exe = get_python_exe()
            
            dependencies = [
                "cryptography>=42.0.0",
                "Pillow>=10.0.0",
                "numpy>=1.24.0",
                "pandas>=2.0.0",
                "scikit-learn>=1.3.0",
                "matplotlib>=3.7.0",
                "tqdm>=4.65.0"
            ]
            
            for dep in dependencies:
                cmd = [
                    python_exe, "-m", "pip", "install",
                    dep,
                    "--trusted-host", "pypi.org",
                    "--trusted-host", "files.pythonhosted.org"
                ]
                
                try:
                    subprocess.run(
                        cmd,
                        capture_output=True,
                        text=True,
                        timeout=60,
                        **get_hide_window_kwargs()
                    )
                except:
                    pass  # Continue even if one fails
            
            messages.addMessage("  ✓ Dependencies lain diinstal")
            return True
            
        except Exception as e:
            messages.addWarningMessage(f"  ⚠️ Error instal dependencies: {str(e)}")
            return False
    
    def test_imports(self):
        """Test importing all required modules"""
        modules_to_test = [
            'torch',
            'torchvision',
            'fastai',
            'cryptography',
            'PIL',
            'numpy',
            'pandas',
            'arcgis.learn'
        ]
        
        results = {}
        for module in modules_to_test:
            try:
                if module == 'PIL':
                    import PIL
                    results[module] = True
                elif module == 'arcgis.learn':
                    import arcgis.learn
                    results[module] = True
                else:
                    importlib.import_module(module)
                    results[module] = True
            except Exception as e:
                results[module] = False
        
        return results

class ManageLandsat8(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "1. Manage Landsat 8 (Buat Mosaic Dataset)"
        self.category = "01. Data Preparation"
        self.description = "Membuat Mosaic Dataset dari data mentah Landsat 8 (Level 1 atau Level 2)."
        self.canRunInBackground = False

    def getParameterInfo(self):
        """Define parameter definitions"""
        
        # Param 0: Output Geodatabase
        param0 = arcpy.Parameter(
            displayName="Output Geodatabase",
            name="out_gdb",
            datatype="DEWorkspace", 
            parameterType="Required",
            direction="Input"
        )
        param0.filter.list = ["Local Database"]
        
        # Set default ke project workspace
        try:
            aprx = arcpy.mp.ArcGISProject("CURRENT")
            param0.value = aprx.defaultGeodatabase
        except:
            pass

        # Param 1: Mosaic Dataset Name
        param1 = arcpy.Parameter(
            displayName="Mosaic Dataset Name",
            name="mosaic_name",
            datatype="GPString",
            parameterType="Required",
            direction="Input"
        )

        # Param 2: Input Source Data (Folder)
        param2 = arcpy.Parameter(
            displayName="Input Source Data (Folder L8)",
            name="in_folder",
            datatype="DEFolder",
            parameterType="Required",
            direction="Input"
        )

        # Param 3: Dataset Type (Level 1 / Level 2)
        param3 = arcpy.Parameter(
            displayName="Dataset Type",
            name="dataset_type",
            datatype="GPString",
            parameterType="Required",
            direction="Input"
        )
        param3.filter.type = "ValueList"
        param3.filter.list = ["Level 1", "Level 2"]
        param3.value = "Level 2"

        # Param 4: Multispectral (Checkbox)
        param4 = arcpy.Parameter(
            displayName="Multispectral",
            name="multispectral",
            datatype="GPBoolean",
            parameterType="Optional",
            direction="Input"
        )
        param4.value = True

        # Param 5: Pansharpen (Checkbox)
        param5 = arcpy.Parameter(
            displayName="Pansharpen",
            name="pansharpen",
            datatype="GPBoolean",
            parameterType="Optional",
            direction="Input"
        )
        param5.value = False

        # Param 6: Panchromatic (Checkbox)
        param6 = arcpy.Parameter(
            displayName="Panchromatic",
            name="panchromatic",
            datatype="GPBoolean",
            parameterType="Optional",
            direction="Input"
        )
        param6.value = False
        
        # Param 7: Output Mosaic Dataset (Derived)
        param7 = arcpy.Parameter(
            displayName="Output Mosaic Dataset",
            name="out_mosaic",
            datatype="DEMosaicDataset",
            parameterType="Derived",
            direction="Output"
        )

        return [param0, param1, param2, param3, param4, param5, param6, param7]

    def isLicensed(self):
        return True

    def updateParameters(self, parameters):
        return

    def updateMessages(self, parameters):
        return

    def execute(self, parameters, messages):
        """Execute the tool"""
        
        # Check License
        SupabaseGuard.check_license()
        
        try:
            out_gdb = parameters[0].valueAsText
            mosaic_name = parameters[1].valueAsText
            in_folder = parameters[2].valueAsText
            dataset_type = parameters[3].valueAsText
            use_multi = parameters[4].value
            use_pan_sharpen = parameters[5].value
            use_pan = parameters[6].value
            
            # Construct full path
            mosaic_full_path = os.path.join(out_gdb, mosaic_name)
            
            messages.addMessage("============================================================")
            messages.addMessage(" MANAGE LANDSAT 8 (MOSAIC DATASET)")
            messages.addMessage("============================================================")
            
            # Check if mosaic already exists
            if arcpy.Exists(mosaic_full_path):
                messages.addErrorMessage(f"\n[ERROR] Mosaic dataset '{mosaic_name}' sudah ada!")
                messages.addErrorMessage("Solusi:")
                messages.addErrorMessage("1. Hapus Mosaic Dataset yang lama di Catalog.")
                messages.addErrorMessage("2. Atau gunakan nama lain untuk Mosaic Dataset baru.")
                messages.addMessage("============================================================")
                raise arcpy.ExecuteError
            
            messages.addMessage(f" Output GDB: {os.path.basename(out_gdb)}")
            messages.addMessage(f" Mosaic Name: {mosaic_name}")
            messages.addMessage(f" Tipe Data: {dataset_type}")
            messages.addMessage("============================================================")
            
            # 1. Create Mosaic Dataset
            messages.addMessage("\n[1] Membuat Mosaic Dataset...")
            sr = arcpy.SpatialReference(4326)  # WGS84
            arcpy.management.CreateMosaicDataset(
                in_workspace=out_gdb,
                in_mosaicdataset_name=mosaic_name,
                coordinate_system=sr,
                num_bands=7 if dataset_type == "Level 2" else 11
            )
            messages.addMessage("✓ Mosaic Dataset berhasil dibuat")
            
            # 2. Find and add Landsat data
            messages.addMessage("\n[2] Menambahkan data Landsat 8...")
            
            # Find MTL file
            mtl_file = self.find_mtl_file(in_folder)
            if not mtl_file:
                messages.addErrorMessage("File MTL.txt tidak ditemukan!")
                raise arcpy.ExecuteError
            
            messages.addMessage(f"✓ File metadata ditemukan: {os.path.basename(mtl_file)}")
            
            # Add rasters to mosaic dataset
            try:
                arcpy.management.AddRastersToMosaicDataset(
                    in_mosaic_dataset=mosaic_full_path,
                    raster_type="Landsat 8",
                    input_path=mtl_file,
                    update_cellsize_ranges="UPDATE_CELL_SIZES",
                    update_boundary="UPDATE_BOUNDARY",
                    update_overviews="NO_OVERVIEWS",
                    maximum_pyramid_levels=-1,
                    maximum_cell_size=0,
                    minimum_dimension=1500,
                    spatial_reference=None,
                    filter=None,
                    sub_folder="NO_SUBFOLDERS",
                    duplicate_items_action="ALLOW_DUPLICATES",
                    build_pyramids="BUILD_PYRAMIDS",
                    calculate_statistics="CALCULATE_STATISTICS",
                    build_thumbnails="NO_THUMBNAILS",
                    operation_description="Added Landsat 8 data",
                    force_spatial_reference="NO_FORCE_SPATIAL_REFERENCE",
                    estimate_statistics="ESTIMATE_STATISTICS"
                )
                
                # Check if data was added
                item_count = int(arcpy.management.GetCount(mosaic_full_path).getOutput(0))
                if item_count == 0:
                    messages.addWarningMessage("Tidak ada data yang ditambahkan, mencoba metode alternatif...")
                    self.add_rasters_alternative(mosaic_full_path, in_folder, messages)
                
                messages.addMessage(f"✓ Berhasil menambahkan {item_count} item ke Mosaic Dataset")
                
            except Exception as e:
                messages.addWarningMessage(f"Gagal menambahkan dengan metode standar: {str(e)}")
                messages.addMessage("Mencoba metode alternatif...")
                self.add_rasters_alternative(mosaic_full_path, in_folder, messages)
            
            # 3. Build statistics and pyramids
            messages.addMessage("\n[3] Membangun statistik dan pyramids...")
            try:
                arcpy.management.BuildPyramidsandStatistics(
                    in_workspace=mosaic_full_path,
                    include_subdirectories="NO_SUBDIRECTORIES",
                    build_pyramids="BUILD_PYRAMIDS",
                    calculate_statistics="CALCULATE_STATISTICS",
                    # BUILD_ON_SOURCE tidak didukung untuk Mosaic Dataset
                    block_field="",
                    estimate_statistics="ESTIMATE_STATISTICS",
                    x_skip_factor="1",
                    y_skip_factor="1",
                    ignore_values="",
                    pyramid_level="-1",
                    skip_first="NONE",
                    resample_technique="NEAREST",
                    compression_type="DEFAULT",
                    compression_quality="75",
                    skip_existing="SKIP_EXISTING"
                )
                messages.addMessage("✓ Statistik dan pyramids berhasil dibangun")
            except:
                messages.addWarningMessage("⚠️ Gagal membangun statistik/pyramids (mungkin sudah dibangun)")
            
            # 4. Set properties
            messages.addMessage("\n[4] Mengatur properti Mosaic Dataset...")
            try:
                # Set processing template
                if use_pan_sharpen:
                    processing_template = "Pansharpen"
                elif use_pan:
                    processing_template = "Panchromatic"
                else:
                    processing_template = "Multispectral"
                
                # Set default rendering
                arcpy.management.SetMosaicDatasetProperties(
                    mosaic_full_path,
                    rows_maximum_imagesize=4096,
                    columns_maximum_imagesize=4096,
                    allowed_compressions="JPEG;LZ77;NONE",
                    default_compression_type="JPEG",
                    JPEG_quality=80,
                    LERC_Tolerance=0.01,
                    resampling_type="NEAREST",
                    clip_to_items="NO_CLIP",
                    processing_templates=None,  # Not setting template directly here as it often conflicts
                    default_processing_template=processing_template
                )
                messages.addMessage(f"✓ Processing template: {processing_template}")
                
            except Exception as e:
                messages.addWarningMessage(f"⚠️ Gagal mengatur properti: {str(e)}")
            
            # Set output parameter
            parameters[7].value = mosaic_full_path
            
            messages.addMessage("")
            messages.addMessage("============================================================")
            messages.addMessage("✅ MOSAIC DATASET BERHASIL DIBUAT!")
            messages.addMessage(f"Path: {mosaic_full_path}")
            messages.addMessage("============================================================")
            
            # Add to map if possible
            try:
                aprx = arcpy.mp.ArcGISProject("CURRENT")
                if aprx.activeMap:
                    aprx.activeMap.addDataFromPath(mosaic_full_path)
                    messages.addMessage("✓ Mosaic Dataset ditambahkan ke peta aktif")
            except:
                pass
            
            return mosaic_full_path
            
        except Exception as e:
            messages.addErrorMessage(f"Error: {str(e)}")
            raise arcpy.ExecuteError

    def find_mtl_file(self, folder_path):
        """Find MTL file in Landsat folder"""
        for root, dirs, files in os.walk(folder_path):
            for file in files:
                if file.upper().endswith("_MTL.TXT"):
                    return os.path.join(root, file)
                elif file.upper().endswith("MTL.txt"):
                    return os.path.join(root, file)
        return None

    def add_rasters_alternative(self, mosaic_path, input_folder, messages):
        """Alternative method: Composite Bands then Add"""
        try:
            # Find all TIF files
            tif_files = []
            for root, dirs, files in os.walk(input_folder):
                for file in files:
                    if file.lower().endswith('.tif'):
                        tif_files.append(os.path.join(root, file))
            
            if not tif_files:
                messages.addErrorMessage("Tidak ada file TIF ditemukan!")
                return
            
            # Map bands
            band_map = {}
            for tif_file in tif_files:
                filename = os.path.basename(tif_file).upper()
                # Check for B1-B11
                for i in range(1, 12):
                    if f"_B{i}.TIF" in filename or f"_SR_B{i}" in filename:
                        band_map[i] = tif_file
            
            if not band_map:
                messages.addErrorMessage("File Band Landsat tidak teridentifikasi (B1-B11).")
                return

            # Sort bands to create composite list
            # Usually Landsat 8 has 11 bands for L1, 7 for L2
            # We try to use as many as found
            sorted_bands = []
            files_to_composite = []
            for i in sorted(band_map.keys()):
                sorted_bands.append(str(i))
                files_to_composite.append(band_map[i])
            
            messages.addMessage(f"  Ditemukan {len(files_to_composite)} band: {', '.join(sorted_bands)}")
            
            if len(files_to_composite) < 3:
                messages.addErrorMessage("Jumlah band terlalu sedikit untuk Composite.")
                return

            # Create Temp Composite
            temp_composite = os.path.join(arcpy.env.scratchFolder, f"L8_Composite_{random.randint(1000,9999)}.tif")
            messages.addMessage("  Membuat Temporary Composite Raster...")
            
            arcpy.management.CompositeBands(files_to_composite, temp_composite)
            
            # Add to Mosaic
            messages.addMessage("  Menambahkan Composite ke Mosaic Dataset...")
            arcpy.management.AddRastersToMosaicDataset(
                mosaic_path,
                raster_type="Raster Dataset",
                input_path=temp_composite,
                update_cellsize_ranges="UPDATE_CELL_SIZES",
                update_boundary="UPDATE_BOUNDARY",
                update_overviews="NO_OVERVIEWS",
                duplicate_items_action="ALLOW_DUPLICATES"
            )
            
            messages.addMessage(f"✓ Berhasil menambahkan data via metode Composite")
            
            # Cleanup temp
            try:
                arcpy.Delete_management(temp_composite)
            except: pass
            
        except Exception as e:
            messages.addErrorMessage(f"Gagal metode alternatif: {str(e)}")
            raise

class ClassifyLandCover(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "2. Klasifikasi Tutupan Lahan (Deep Learning)"
        self.category = "02. Analysis & AI"
        self.description = "Melakukan klasifikasi piksel menggunakan model Deep Learning (.dlpk) yang sudah dilatih."
        self.canRunInBackground = False

    def getParameterInfo(self):
        """Define parameter definitions"""
        
        # Param 0: Input Raster
        param0 = arcpy.Parameter(
            displayName="Input Raster (Layer/Mosaic Dataset)",
            name="in_raster",
            datatype=["GPCompositeLayer", "GPRasterLayer", "DEMosaicDataset", "DERasterDataset"],
            parameterType="Required",
            direction="Input"
        )
        
        # Param 1: Model Definition
        param1 = arcpy.Parameter(
            displayName="File Model Deep Learning (.dlpk atau .emd)",
            name="in_model_definition",
            datatype="DEFile",
            parameterType="Required",
            direction="Input"
        )
        param1.filter.list = ['dlpk', 'emd']

        # Param 2: Output Raster
        param2 = arcpy.Parameter(
            displayName="Output Raster Klasifikasi",
            name="out_classified_raster",
            datatype="DERasterDataset",
            parameterType="Required",
            direction="Output"
        )
        
        # --- Advanced Parameters ---
        
        # --- Advanced Parameters ---
        
        # Param 3: Processing Mode
        param3 = arcpy.Parameter(
            displayName="Processing Mode",
            name="processing_mode",
            datatype="GPString",
            parameterType="Optional",
            direction="Input"
        )
        param3.filter.type = "ValueList"
        param3.filter.list = ["PROCESS_AS_MOSAICKED_IMAGE", "PROCESS_AS_RASTER", "PROCESS_ITEMS_SEPARATELY"]
        param3.value = "PROCESS_AS_MOSAICKED_IMAGE"
        
        # Param 4: Padding
        param4 = arcpy.Parameter(
            displayName="Padding",
            name="padding",
            datatype="GPLong",
            parameterType="Optional",
            direction="Input"
        )
        param4.value = 64
        
        # Param 5: Batch Size
        param5 = arcpy.Parameter(
            displayName="Batch Size",
            name="batch_size",
            datatype="GPLong",
            parameterType="Optional",
            direction="Input"
        )
        param5.value = 4
        
        # Param 6: Threshold
        param6 = arcpy.Parameter(
            displayName="Threshold (Confidence)",
            name="threshold",
            datatype="GPDouble",
            parameterType="Optional",
            direction="Input"
        )
        param6.value = 0.5
        
        # Param 7: Predict Background
        param7 = arcpy.Parameter(
            displayName="Predict Background",
            name="predict_background",
            datatype="GPBoolean",
            parameterType="Optional",
            direction="Input"
        )
        param7.value = True
        
        # Param 8: Test Time Augmentation
        param8 = arcpy.Parameter(
            displayName="Test Time Augmentation",
            name="test_time_augmentation",
            datatype="GPBoolean",
            parameterType="Optional",
            direction="Input"
        )
        param8.value = True
        
        # Param 9: Tile Size
        param9 = arcpy.Parameter(
            displayName="Tile Size",
            name="tile_size",
            datatype="GPLong",
            parameterType="Optional",
            direction="Input"
        )
        param9.value = 512
        
        # Param 10: Processor Type
        param10 = arcpy.Parameter(
            displayName="Processor Type",
            name="processor_type",
            datatype="GPString",
            parameterType="Optional",
            direction="Input"
        )
        param10.filter.type = "ValueList"
        param10.filter.list = ["GPU", "CPU"]
        param10.value = "GPU"
        
        # Param 11: Use only CPU cores
        param11 = arcpy.Parameter(
            displayName="Number of CPU Cores",
            name="cpu_cores",
            datatype="GPLong",
            parameterType="Optional",
            direction="Input"
        )
        param11.value = 4
        
        # --- Polygon Conversion Parameters ---
        
        # Param 12: Convert to Polygon
        param12 = arcpy.Parameter(
            displayName="Convert to Polygon (Vector)",
            name="convert_to_polygon",
            datatype="GPBoolean",
            parameterType="Optional",
            direction="Input"
        )
        param12.value = True
        
        # Param 13: Simplify Polygon
        param13 = arcpy.Parameter(
            displayName="Simplify Polygon (Smooth Edges)",
            name="simplify_polygon",
            datatype="GPBoolean",
            parameterType="Optional",
            direction="Input"
        )
        param13.value = True
        
        # Param 14: Output Standard (SNI)
        param14 = arcpy.Parameter(
            displayName="Output Classification Standard",
            name="output_standard",
            datatype="GPString",
            parameterType="Optional",
            direction="Input"
        )
        param14.filter.type = "ValueList"
        param14.filter.list = ["Original (Esri)", "SNI Indonesia (1:50.000)"]
        param14.value = "SNI Indonesia (1:50.000)"

        return [param0, param1, param2, param3, param4, param5, param6, param7, 
                param8, param9, param10, param11, param12, param13, param14]

    def isLicensed(self):
        """Check for required licenses"""
        try:
            if arcpy.CheckExtension("ImageAnalyst") == "Available":
                return True
            if arcpy.CheckExtension("Spatial") == "Available":
                return True
        except:
            pass
        return False

    def updateParameters(self, parameters):
        """Modify parameters"""
        return

    def updateMessages(self, parameters):
        """Modify messages"""
        # Check license
        if not self.isLicensed():
            parameters[0].setErrorMessage(
                "Tool ini membutuhkan lisensi Image Analyst atau Spatial Analyst."
            )
        
        # Check if model file exists
        if parameters[1].valueAsText:
            model_path = parameters[1].valueAsText
            if not os.path.exists(model_path):
                parameters[1].setErrorMessage("File model tidak ditemukan!")
        return

    def execute(self, parameters, messages):
        """Execute the tool"""
        
        # Check License
        SupabaseGuard.check_license()
        
        try:
            # Get parameters
            # Get parameters
            in_raster = parameters[0].valueAsText
            in_model = parameters[1].valueAsText
            out_raster = parameters[2].valueAsText
            
            # --- BAND VALIDATION (LEVEL 2 ENFORCEMENT) ---
            desc_raster = arcpy.Describe(in_raster)
            if hasattr(desc_raster, "bandCount"):
                if desc_raster.bandCount > 8:
                    messages.addErrorMessage("\n⛔ ERROR: DATA TIDAK VALID (LEVEL 1 TERDETEKSI) ⛔")
                    messages.addErrorMessage("Tool ini dikunci hanya untuk data Landsat 8 Level 2 (Surface Reflectance).")
                    messages.addErrorMessage(f"Input Anda memiliki {desc_raster.bandCount} band (Indikasi Level 1).")
                    messages.addErrorMessage("Solusi:")
                    messages.addErrorMessage("1. Gunakan data Landsat 8 Level 2 (Download 'LC08_L2SP...').")
                    messages.addErrorMessage("2. Atau gunakan tool 'Manage Landsat 8' dengan tipe 'Level 2' (jika data sumbernya memang SR).")
                    raise arcpy.ExecuteError
            # ---------------------------------------------
            
            # HARDCODED: Always treat as Level 2
            landsat_level = 2
            
            # Advanced parameters (Indices shifted because Index 3 was removed)
            proc_mode = parameters[3].valueAsText
            padding = parameters[4].value
            batch_size = parameters[5].value
            threshold = parameters[6].value
            predict_bg = parameters[7].value
            tta = parameters[8].value
            tile_size = parameters[9].value
            processor_type = parameters[10].valueAsText
            cpu_cores = parameters[11].value
            convert_to_polygon = parameters[12].value
            simplify_polygon = parameters[13].value
            output_standard = parameters[14].valueAsText
            
            messages.addMessage("=" * 60)
            messages.addMessage("DEEP LEARNING LAND COVER CLASSIFICATION")
            messages.addMessage("=" * 60)
            
            # ====================================================================
            # 1. PRE-CHECK DEPENDENCIES
            # ====================================================================
            messages.addMessage("\n[1] MEMERIKSA DEPENDENCIES...")
            
            deps_ok = self.check_dependencies_preflight(messages)
            if not deps_ok:
                messages.addErrorMessage("\n❌ DEPENDENCIES TIDAK LENGKAP!")
                messages.addErrorMessage("Jalankan tool '0. Install Deep Learning Libraries' terlebih dahulu.")
                raise arcpy.ExecuteError
            
            messages.addMessage("✓ Semua dependencies OK")
            
            # ====================================================================
            # 2. CHECK LICENSES
            # ====================================================================
            messages.addMessage("\n[2] MEMERIKSA LISENSI...")
            
            license_checked = False
            if arcpy.CheckExtension("ImageAnalyst") == "Available":
                arcpy.CheckOutExtension("ImageAnalyst")
                messages.addMessage("✓ Menggunakan lisensi Image Analyst")
                license_checked = True
            elif arcpy.CheckExtension("Spatial") == "Available":
                arcpy.CheckOutExtension("Spatial")
                messages.addMessage("✓ Menggunakan lisensi Spatial Analyst")
                license_checked = True
            
            if not license_checked:
                messages.addErrorMessage("❌ Tidak ada lisensi yang tersedia!")
                raise arcpy.ExecuteError
            
            # ====================================================================
            # 3. VALIDATE INPUTS
            # ====================================================================
            messages.addMessage("\n[3] VALIDASI INPUT...")
            
            # Check if input raster exists
            if not arcpy.Exists(in_raster):
                messages.addErrorMessage(f"❌ Input raster tidak ditemukan: {in_raster}")
                raise arcpy.ExecuteError
            
            # Check if model file exists
            if not os.path.exists(in_model):
                messages.addErrorMessage(f"❌ File model tidak ditemukan: {in_model}")
                raise arcpy.ExecuteError
            
            messages.addMessage(f"✓ Input raster: {os.path.basename(in_raster)}")
            messages.addMessage(f"✓ Model: {os.path.basename(in_model)}")
            messages.addMessage(f"✓ Output: {out_raster}")
            
            # ====================================================================
            # 4. PREPARE ARGUMENTS
            # ====================================================================
            messages.addMessage("\n[4] MENYIAPKAN PARAMETER...")
            
            # Build arguments string
            args_dict = {
                'padding': padding,
                'batch_size': batch_size,
                'threshold': threshold,
                'predict_background': str(predict_bg).lower() if isinstance(predict_bg, bool) else predict_bg,
                'test_time_augmentation': str(tta).lower() if isinstance(tta, bool) else tta,
                'tile_size': tile_size,
                'landsat_imagery_level': landsat_level,
                'processor_type': processor_type.lower()
            }
            
            # Filter out None values
            args_dict = {k: v for k, v in args_dict.items() if v is not None}
            
            # Convert to string format
            args_list = [f"{k} {v}" for k, v in args_dict.items()]
            args_str = ";".join(args_list)
            
            messages.addMessage(f"Arguments: {args_str}")
            
            # ====================================================================
            # 5. SET ENVIRONMENT
            # ====================================================================
            messages.addMessage("\n[5] MENGATUR ENVIRONMENT...")
            
            # Save current environment
            old_workspace = arcpy.env.workspace
            old_cellSize = arcpy.env.cellSize
            old_extent = arcpy.env.extent
            old_scratch = arcpy.env.scratchWorkspace
            
            try:
                # Set environment for deep learning
                arcpy.env.workspace = os.path.dirname(out_raster)
                arcpy.env.cellSize = in_raster
                arcpy.env.extent = in_raster
                arcpy.env.scratchWorkspace = arcpy.env.scratchFolder if hasattr(arcpy.env, 'scratchFolder') else old_scratch
                
                # Set processor type
                if processor_type == "GPU":
                    arcpy.env.processorType = "GPU"
                    messages.addMessage("✓ Mode: GPU (dengan CUDA jika tersedia)")
                else:
                    arcpy.env.processorType = "CPU"
                    arcpy.env.parallelProcessingFactor = str(cpu_cores)
                    messages.addMessage(f"✓ Mode: CPU ({cpu_cores} cores)")
                
            except:
                messages.addWarningMessage("⚠️ Gagal mengatur environment, menggunakan default")
            
            # ====================================================================
            # 6. RUN CLASSIFICATION
            # ====================================================================
            messages.addMessage("\n[6] MENJALANKAN KLASIFIKASI...")
            messages.addMessage("Mohon tunggu, proses ini mungkin memakan waktu...")
            
            start_time = time.time()
            
            try:
                # Ensure output path is safe (handle locks)
                if arcpy.Exists(out_raster):
                    try:
                        arcpy.Delete_management(out_raster)
                        messages.addMessage(f"  File lama dihapus: {out_raster}")
                    except:
                        # If delete fails, generate unique name
                        base, ext = os.path.splitext(out_raster)
                        timestamp = time.strftime("%Y%m%d_%H%M%S")
                        out_raster = f"{base}_{timestamp}{ext}"
                        messages.addWarningMessage(f"⚠️ File terkunci, output dialihkan ke: {out_raster}")

                # Enable overwrite
                arcpy.env.overwriteOutput = True

                try:
                    # Run classification using arcpy.ia (Primary Method for Image Analyst)
                    messages.addMessage(f"  Menggunakan metode utama (arcpy.ia)...")
                    
                    output = arcpy.ia.ClassifyPixelsUsingDeepLearning(
                        in_raster=in_raster,
                        in_model_definition=in_model,
                        arguments=args_str,
                        processing_mode=proc_mode
                    )
                    
                    # Save result
                    output.save(out_raster)
                    
                    elapsed = time.time() - start_time
                    messages.addMessage(f"✓ Klasifikasi selesai dalam {elapsed:.1f} detik")
                    messages.addMessage(f"✓ Hasil disimpan di: {out_raster}")
                    
                except Exception as e:
                    messages.addWarningMessage(f"⚠️ Metode utama (arcpy.ia) gagal: {str(e)}")
                    messages.addMessage("Mencoba metode fallback (subprocess)...")
                    
                    try:
                         # Last resort: Run in subprocess to isolate environment
                        self.run_classification_alternative(
                             in_raster, in_model, args_str, proc_mode, messages
                        )
                    except Exception as sub_e:
                        raise e

            except Exception as e:
                # If everything fails, raise the original error for clarity
                raise e

            # ====================================================================
            # 7. POST-PROCESSING
            # ====================================================================
            messages.addMessage("\n[7] POST-PROCESSING...")
            
            try:
                # Build statistics
                arcpy.management.CalculateStatistics(out_raster)
                messages.addMessage("✓ Statistik dihitung")
                
                # Build pyramids
                arcpy.management.BuildPyramids(out_raster)
                messages.addMessage("✓ Pyramids dibangun")
                
            except:
                messages.addWarningMessage("⚠️ Gagal post-processing (tidak fatal)")
            
            # ====================================================================
            # 7. CONVERT TO POLYGON (OPTIONAL)
            # ====================================================================
            if convert_to_polygon:
                messages.addMessage("\n[7] KONVERSI KE POLYGON...")
                try:
                    # Determine output name (Handle GDB vs Shapefile)
                    out_dir = os.path.dirname(out_raster)
                    out_base = os.path.splitext(os.path.basename(out_raster))[0]
                    
                    if out_dir.endswith(".gdb") or ".gdb" in out_dir.lower():
                        # Output is inside a Geodatabase -> Feature Class
                        # GDB Feature Classes cannot contain "." or other special chars (except _)
                        out_base_sanitized = out_base.replace(".", "_").replace("-", "_")
                        poly_name = os.path.join(out_dir, f"{out_base_sanitized}_Poly")
                    else:
                        # Output is in a folder -> Shapefile
                        poly_name = os.path.join(out_dir, f"{out_base}_Poly.shp")
                    
                    if arcpy.Exists(poly_name):
                        arcpy.Delete_management(poly_name)
                    
                    simplify_opt = "SIMPLIFY" if simplify_polygon else "NO_SIMPLIFY"
                    messages.addMessage(f"  Mengkonversi raster ke polygon ({simplify_opt})...")
                    
                    arcpy.conversion.RasterToPolygon(out_raster, poly_name, simplify_opt, "Value")
                    messages.addMessage(f"  ✓ Polygon tercipta: {poly_name}")
                    
                    # Add Fields
                    messages.addMessage("  Menambahkan atribut SNI...")
                    arcpy.management.AddField(poly_name, "Class_Name", "TEXT", field_length=100)
                    arcpy.management.AddField(poly_name, "SNI_Class", "TEXT", field_length=100)
                    arcpy.management.AddField(poly_name, "Luas_Ha", "DOUBLE")
                    
                    # Calculate Fields
                    # Define mapping dictionaries (Updated to NLCD Codes based on user feedback)
                    # Codes: 11(Water), 21-24(Built), 31(Barren), 41-43(Forest), 52(Shrub), 71(Grass), 81(Pasture), 82(Crops), 90/95(Wetlands)
                    esri_classes = {
                        11: "Open Water",
                        12: "Perennial Snow/Ice",
                        21: "Developed, Open Space",
                        22: "Developed, Low Intensity",
                        23: "Developed, Medium Intensity",
                        24: "Developed, High Intensity",
                        31: "Barren Land",
                        41: "Deciduous Forest",
                        42: "Evergreen Forest",
                        43: "Mixed Forest",
                        52: "Shrub/Scrub",
                        71: "Herbaceous",
                        81: "Hay/Pasture",
                        82: "Cultivated Crops",
                        90: "Woody Wetlands",
                        95: "Emergent Herbaceous Wetlands"
                    }
                    
                    sni_mapping = {
                        11: "Air",                            # Open Water -> Air
                        12: "Lainnya",                        # Ice/Snow -> Lainnya
                        21: "Area Terbangun",                 # Dev Open -> Area Terbangun
                        22: "Area Terbangun",                 # Dev Low -> Area Terbangun
                        23: "Area Terbangun",                 # Dev Med -> Area Terbangun
                        24: "Area Terbangun",                 # Dev High -> Area Terbangun
                        31: "Area Terbuka",                   # Barren -> Area Terbuka
                        41: "Hutan",                          # Deciduous -> Hutan
                        42: "Hutan",                          # Evergreen -> Hutan
                        43: "Hutan",                          # Mixed -> Hutan
                        52: "Area Terbuka",                   # Shrub -> Area Terbuka
                        71: "Area Terbuka",                   # Grassland -> Area Terbuka
                        81: "Pertanian (Sawah/Ladang)",       # Pasture -> Pertanian
                        82: "Pertanian (Sawah/Ladang)",       # Crops -> Pertanian
                        90: "Rawa",                           # Wetlands -> Rawa
                        95: "Rawa"                            # Wetlands -> Rawa
                    }
                    
                    with arcpy.da.UpdateCursor(poly_name, ["gridcode", "Class_Name", "SNI_Class"]) as cursor:
                        for row in cursor:
                            grid_val = row[0]
                            
                            # 1. Class Name (Esri/NLCD)
                            row[1] = esri_classes.get(grid_val, f"Unknown ({grid_val})")
                            
                            # 2. SNI Class
                            if output_standard.startswith("SNI"):
                                row[2] = sni_mapping.get(grid_val, "Lainnya")
                            else:
                                row[2] = row[1] # Keep original if not SNI
                            
                            cursor.updateRow(row)
                    
                    messages.addMessage("  ✓ Atribut Kelas SNI terisi")
                            
                    # 3. Calculate Geometry Attributes (Geodesic Area used for WGS84/GCS correctness)
                    messages.addMessage("  Menghitung luas (Hectares)...")
                    arcpy.management.CalculateGeometryAttributes(
                        in_features=poly_name,
                        geometry_property=[["Luas_Ha", "AREA_GEODESIC"]],
                        area_unit="HECTARES"
                    )
                    messages.addMessage("  ✓ Luas berhasil dihitung")
                    
                    # Add to map
                    try:
                        aprx = arcpy.mp.ArcGISProject("CURRENT")
                        if aprx.activeMap:
                            aprx.activeMap.addDataFromPath(poly_name)
                    except:
                        pass
                        
                except Exception as e:
                    messages.addWarningMessage(f"⚠️ Gagal konversi polygon: {str(e)}")

            # ====================================================================
            # 8. CLEANUP
            # ====================================================================
            messages.addMessage("\n[8] CLEANUP...")
            
            # Restore environment
            arcpy.env.workspace = old_workspace
            arcpy.env.cellSize = old_cellSize
            arcpy.env.extent = old_extent
            arcpy.env.scratchWorkspace = old_scratch
            
            # Check in license
            arcpy.CheckInExtension("ImageAnalyst")
            arcpy.CheckInExtension("Spatial")
            
            # Add to map if possible
            try:
                aprx = arcpy.mp.ArcGISProject("CURRENT")
                if aprx.activeMap:
                    aprx.activeMap.addDataFromPath(out_raster)
                    messages.addMessage("✓ Hasil ditambahkan ke peta aktif")
            except:
                pass
            
            # ====================================================================
            # 9. FINAL REPORT
            # ====================================================================
            messages.addMessage("\n" + "=" * 80)
            messages.addMessage("✅ KLASIFIKASI BERHASIL!")
            messages.addMessage("=" * 80)
            
            return out_raster

        except Exception as outer_e:
            messages.addErrorMessage(f"❌ Gagal menjalankan klasifikasi: {str(outer_e)}")
            # Cleanup on error
            try:
                arcpy.CheckInExtension("ImageAnalyst")
                arcpy.CheckInExtension("Spatial")
            except:
                pass
            
            raise arcpy.ExecuteError
    
    def check_dependencies_preflight(self, messages):
        """Quick check for required dependencies"""
        required = ['torch', 'torchvision', 'fastai', 'cryptography']
        
        for dep in required:
            try:
                if dep == 'torch':
                    import torch
                    messages.addMessage(f"  ✓ PyTorch: {torch.__version__}")
                    # Check CUDA
                    if torch.cuda.is_available():
                        messages.addMessage(f"    CUDA: {torch.version.cuda}")
                elif dep == 'torchvision':
                    import torchvision
                    messages.addMessage(f"  ✓ TorchVision: {torchvision.__version__}")
                elif dep == 'fastai':
                    import fastai
                    messages.addMessage(f"  ✓ fastai: {fastai.__version__}")
                elif dep == 'cryptography':
                    import cryptography
                    messages.addMessage(f"  ✓ cryptography: {cryptography.__version__}")
            except ImportError:
                messages.addErrorMessage(f"  ✗ {dep}: MISSING")
                return False
            except Exception as e:
                messages.addWarningMessage(f"  ⚠️ {dep}: {str(e)}")
        
        return True
    
    def run_classification_alternative(self, in_raster, in_model, args_str, proc_mode, messages):
        """Alternative classification method"""
        # Try different approaches
        
        # Approach 1: Use arcpy.gp directly
        try:
            messages.addMessage("  Mencoba approach 1: arcpy.gp...")
            output = arcpy.gp.ClassifyPixelsUsingDeepLearning(
                in_raster, in_model, None, args_str, proc_mode
            )
            return output
        except:
            pass
        
        # Approach 2: Use subprocess
        try:
            messages.addMessage("  Mencoba approach 2: subprocess...")
            
            # Build command
            # Build command
            cmd = [
                get_python_exe(), "-c",
                f"""
import arcpy
import os
try:
    arcpy.CheckOutExtension('ImageAnalyst')
    arcpy.env.overwriteOutput = True
    print('Starting classification...')
    output = arcpy.ia.ClassifyPixelsUsingDeepLearning(
        r'{in_raster}', r'{in_model}', arguments='{args_str}', processing_mode='{proc_mode}'
    )
    print('Saving to {out_raster}...')
    output.save(r'{out_raster}')
    print('SUCCESS')
except Exception as e:
    print(f'FAILURE: {{str(e)}}')
"""
            ]
            
            result = subprocess.run(cmd, capture_output=True, text=True, **get_hide_window_kwargs())
            
            if "SUCCESS" in result.stdout:
                # Return the raster object from the saved path
                return arcpy.Raster(out_raster)
            else:
                messages.addErrorMessage(f"Subprocess output: {result.stdout}")
                messages.addErrorMessage(f"Subprocess error: {result.stderr}")
        except:
            pass
        
        raise Exception("Semua metode alternatif gagal")
